﻿#if NUNIT
#region References

using System;
using System.Collections;
using System.Text;
using gov.va.med.vbecs.Common.AppServices;
using NUnit.Framework;

using gov.va.med.vbecs.UnitTests;

#endregion

namespace gov.va.med.vbecs.HL7Server.UnitTests
{
    [TestFixture]
    public class HL7Server_Server : BaseTest
    {
        private string segName = string.Empty;
        private const string PID = "PID";
        private const string MRG = "MRG";

        private const string CPRS_MESSAGE =
            "MSH|^~\\&|OERR|589|VBECS|589|20080526135401-0500||OMG^O19|5894312166|T|2.4|||AL|AL|USA" +
            CARRIAGE_RETURN +
            "PID|||5890045694V193464^^^^NI~493013508^^^^SS~14161^^^^PI||LASTNAME^FIRSTNAME^MIDDLENAME||19601126|M" +
            CARRIAGE_RETURN +
            "PV1||O|KC-1010 NEW PT CLINIC 2|||||||" +
            CARRIAGE_RETURN +
            "ORC|NW|22179410^OR||22179406^OR|||||200805261354-0500|53342^USER^ONE||53342^USER^ONE||||^Transfuse|589^VA HEARTLAND - WEST, VISN 15|||" +
            CARRIAGE_RETURN +
            "OBR|1|22179410^OR||1^TYPE \\T\\ SCREEN^99VBC^^|||||||O||89932;3080526;4;6478||^^SEND PATIENT||||||||||||^^^20080526^200805261354-0500^R^^" +
            CARRIAGE_RETURN +
            "OBX|1||191^ALANINE AMINOTRANSFERASE^99LRT||17|IU/L|8-40||||F|||20020114093059-0500" +
            CARRIAGE_RETURN +
            "DG1|1||^^^^^|||A " +
            CARRIAGE_RETURN;

        private const string PATIENT_MERGE_MESSAGE =
            "MSH^~|\\&^MPIF TRIGGER^^VBECS ADT^589^20110410173117-0500^^ADT~A40^5894298240^T^2.4^^^AL^AL^" +
            CARRIAGE_RETURN +
            "EVN^^^^^53342~USER~TEST~~~~~~USVHA&&0363~L~~~NI~VA FACILITY ID&589&L^^589" +
            CARRIAGE_RETURN +
            "PID^1^1000045036V498441~~~USVHA&&0363~NI~VA FACILITY ID&200M&L^1000045036V498441~~~USVHA&&0363~NI~VA FACILITY ID&200M&L|321000004~~~USSSA&&0363~SS~VA FACILITY ID&589&L|412583~~~USVHA&&0363~PI~VA FACILITY ID&589&L|321000003~~~USSSA&&0363~SS~VA FACILITY ID&589&L~~20110410|321000002~~~USSSA&&0363~SS~VA FACILITY ID&589&L~~20110410|5890045036V502660~~~USVHA&&0363~NI~VA FACILITY ID&589&L~~20061213|5890045035V840731~~~USVHA&&0363~NI~VA FACILITY ID&589&L~~20061213|1000045035V840731~~~USVHA&&0363~NI~VA FACILITY ID&200M&L~~20061213|5890045034V975882~~~USVHA&&0363~NI~VA FACILITY ID&589&L~~20061213|1000045034V103607~~~USVHA&&0363~NI~VA FACILITY ID&200M&L~~20110310^^LASTNAME~FIRSTNAME~MIDDLENAME^DOFKDP~~~~~~M^19230101^F^^\"\"^22 WEST ST~\"\"~MUNDELEIN~IL~60060~USA~P~\"\"~097|~~CHICAGO~IL~~~N^097^(111)555-8332~PRN~PH|(630)456-8484~WPN~PH^(111)555-8484^^\"\"^0^^321000004^^^\"\"^CHICAGO IL^N^^^^^\"\"^^" +
            CARRIAGE_RETURN +
            "PD1^^^VA HEARTLAND - WEST, VISN 15~D~589" +
            CARRIAGE_RETURN +
            "MRG^1000045116V065714~~~USVHA&&0363~NI~VA FACILITY ID&589&L|412662~~~USVHA&&0363~PI~VA FACILITY ID&589&L^^^^^^LASTNAME~FIRSTNAME~MIDDLENAME" +
            CARRIAGE_RETURN;

        //Constants match  ValidateVbecsRequiredMessageContent()
        const string detailedErrorMessageKey = "DetailedErrorMessage";
        const string acknowledgementErrorMessageKey = "AcknowledgementErrorMessage";
        //Constants match ValidatePatientName()
        const string patientNameNotFound = "The Patient's Last Name or First Name was not found in the HL7 Message.";
        const string patientNameFieldSizeNotSupported = "The Patient's Name field size exceeds VBECS maximum supported value.";

        [SetUp]
        protected void SetUp()
        {
            base.BaseSetUp();

            if (this.RefreshData)
            {
                this.RefreshData = false;
            }
        }

        const string NameComponentTooBig = "The Patient's {0} Name field size ({1}) exceeds VBECS maximum supported value ({2} characters).";
        const string FullNameTooBig = "\nThe Patient's Full Name field size ({0}) exceeds VBECS maximum supported value ({1} characters).";
        /// <summary>
        /// Code reuse
        /// </summary>
        /// <param name="nameOrdinal"></param>
        /// <param name="maxLength"></param>
        /// <param name="actualLength"></param>
        /// <param name="segment"></param>
        /// <param name="errTable"></param>
        private void AssertMaxLengthChecks(string nameOrdinal, int maxLength, int actualLength, string segment, Hashtable errTable)
        {
            StringBuilder errMsg = new StringBuilder();
            errMsg.Append(string.Format(NameComponentTooBig, nameOrdinal, actualLength, maxLength));

            if (nameOrdinal != "Full")
            {
                int fullnameLen = 0;
                if (nameOrdinal == "First") fullnameLen = actualLength + 20;        //LASTNAME+MIDDLENAME + 2(comma,space)
                if (nameOrdinal == "Middle") fullnameLen = actualLength + 19;        //LASTNAME+FIRSTNAME + 2 (comma,space)
                if (nameOrdinal == "Last") fullnameLen = actualLength + 21;          //FIRSTNAME+MIDDLENAME + 2(comma,space)

                errMsg.Append(string.Format(FullNameTooBig, fullnameLen, maxLength));
            }

            Assert.AreEqual(2, errTable.Count, nameOrdinal + "name too long in " + segment + " - error count");
            Assert.AreEqual(patientNameFieldSizeNotSupported, errTable[acknowledgementErrorMessageKey].ToString(), nameOrdinal + "name too long in " + segment + "  - acknowledgment");
            Assert.AreEqual(errMsg.ToString(), errTable[detailedErrorMessageKey].ToString(), nameOrdinal + "name too long in " + segment + "  - detailed error");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_AllGood()
        {
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE);
            Assert.IsNull(tst1, "Valid PID");

            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE);
            Assert.IsNull(tst2, "Valid MRG");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_BlankFirstName()
        {
            segName = PID;
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("FIRSTNAME", string.Empty));
            Assert.AreEqual(2, tst1.Count, segName + "-error count - BlankFirstName");
            Assert.AreEqual(patientNameNotFound, tst1[acknowledgementErrorMessageKey].ToString(), segName + " BlankFirstName - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst1[detailedErrorMessageKey].ToString(), segName + " BlankFirstName - detailed error");
            
            segName = MRG;
            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("FIRSTNAME", string.Empty));
            Assert.AreEqual(2, tst2.Count, segName + "-error count - BlankFirstName");
            Assert.AreEqual(patientNameNotFound, tst2[acknowledgementErrorMessageKey].ToString(), segName + " BlankFirstName - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst2[detailedErrorMessageKey].ToString(), segName + " BlankFirstName - detailed error");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_BlankLastName()
        {
            segName = PID;
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("LASTNAME", string.Empty));
            Assert.AreEqual(2, tst1.Count, segName + "-error count - BlankLastName");
            Assert.AreEqual(patientNameNotFound, tst1[acknowledgementErrorMessageKey].ToString(), segName + " BlankLastName - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst1[detailedErrorMessageKey].ToString(), segName + " BlankLastName - detailed error");

            segName = MRG;
            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("LASTNAME", string.Empty));
            Assert.AreEqual(2, tst2.Count, segName + "-error count - BlankLastName");
            Assert.AreEqual(patientNameNotFound, tst2[acknowledgementErrorMessageKey].ToString(), segName + " BlankLastName - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst2[detailedErrorMessageKey].ToString(), segName + " BlankLastName - detailed error");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_BlankMiddleName()
        {
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("MIDDLENAME", string.Empty));
            Assert.IsNull(tst1, "BlankMiddleName in PID");

            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("MIDDLENAME", string.Empty));
            Assert.IsNull(tst2, "BlankMiddleName in MRG");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_LastNameTooLong()
        {
            var max = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientLastNameMaximumLength", typeof(int));

            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("LASTNAME", "L".PadRight(max+1,'L')));
            AssertMaxLengthChecks("Last", max, max + 1, PID, tst1);

            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("LASTNAME", "L".PadRight(max+1, 'L')));
            AssertMaxLengthChecks("Last", max, max + 1, MRG, tst2);
        }


        [Test]
        public void ValidateVbecsRequiredMessageContent_FirstNameTooLong()
        {
            var max = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientFirstNameMaximumLength", typeof(int));

            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("FIRSTNAME", "F".PadRight(max+1, 'F')));
            AssertMaxLengthChecks("First", max, max + 1, PID, tst1);

            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("FIRSTNAME", "F".PadRight(max+1, 'F')));
            AssertMaxLengthChecks("First", max, max + 1, MRG, tst2);
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_MiddleNameTooLong()
        {
            var max = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientMiddleNameMaximumLength", typeof(int));

            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("MIDDLENAME", "M".PadRight(max+1, 'M')));
            AssertMaxLengthChecks("Middle", max, max + 1, PID, tst1);

            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("MIDDLENAME", "M".PadRight(max+1, 'M')));
            AssertMaxLengthChecks("Middle", max, max + 1, MRG, tst2);
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_FullNameTooLong()
        {
            var max = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientFullNameMaximumLength", typeof(int));
            var div = (max / 3) + 2; //also added to "full name" is comma and space, so I'm adding 2 for overkill

            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("FIRSTNAME", "F".PadRight(div, 'F')).Replace("MIDDLENAME", "M".PadRight(div, 'M')).Replace("LASTNAME", "L".PadRight(div, 'L')));
            AssertMaxLengthChecks("Full", max, max + 8, PID, tst1);     //currently this is 12*3, + 2 --> 12 = max(30)/3 + 2 

            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("FIRSTNAME", "F".PadRight(div, 'F')).Replace("MIDDLENAME", "M".PadRight(div, 'M')).Replace("LASTNAME", "L".PadRight(div, 'L')));
            AssertMaxLengthChecks("Full", max, max + 8, MRG, tst2);     //currently this is 12*3, + 2 --> 12 = max(30)/3 + 2 
        }


        [Test]
        public void ValidateVbecsRequiredMessageContent_BlankNames()
        {
            segName = PID;
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("FIRSTNAME", string.Empty).Replace("LASTNAME", string.Empty).Replace("MIDDLENAME", string.Empty));
            Assert.AreEqual(2, tst1.Count, segName + "-error count - BlankNames");
            Assert.AreEqual(patientNameNotFound, tst1[acknowledgementErrorMessageKey].ToString(), segName + "  BlankNames- acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst1[detailedErrorMessageKey].ToString(), segName + "  BlankNames- detailed error");

            segName = MRG;
            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("FIRSTNAME", string.Empty).Replace("LASTNAME", string.Empty).Replace("MIDDLENAME", string.Empty));
            Assert.AreEqual(2, tst2.Count, segName + "-error count - BlankNames");
            Assert.AreEqual(patientNameNotFound, tst2[acknowledgementErrorMessageKey].ToString(), segName + "  BlankNames- acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst2[detailedErrorMessageKey].ToString(), segName + "  BlankNames- detailed error");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_NoNames()
        {
            segName = PID;
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("LASTNAME^FIRSTNAME^MIDDLENAME", string.Empty));
            Assert.AreEqual(2, tst1.Count, segName + "-error count - NoNames");
            Assert.AreEqual(patientNameNotFound, tst1[acknowledgementErrorMessageKey].ToString(), segName + " NoNames - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst1[detailedErrorMessageKey].ToString(), segName + " NoNames - detailed error");

            segName = MRG;
            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("LASTNAME~FIRSTNAME~MIDDLENAME", string.Empty));
            Assert.AreEqual(2, tst2.Count, segName + "-error count - NoNames");
            Assert.AreEqual(patientNameNotFound, tst2[acknowledgementErrorMessageKey].ToString(), segName + " NoNames - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst2[detailedErrorMessageKey].ToString(), segName + " NoNames - detailed error");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_OnlyOneName()
        {
            segName = PID;
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("^FIRSTNAME^MIDDLENAME", string.Empty));
            Assert.AreEqual(2, tst1.Count, segName + "-error count - OnlyOneName");
            Assert.AreEqual(patientNameNotFound, tst1[acknowledgementErrorMessageKey].ToString(), segName + " OnlyOneName - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst1[detailedErrorMessageKey].ToString(), segName + " OnlyOneName - detailed error");

            segName = MRG;
            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("~FIRSTNAME~MIDDLENAME", string.Empty));
            Assert.AreEqual(2, tst2.Count, segName + "-error count - OnlyOneName");
            Assert.AreEqual(patientNameNotFound, tst2[acknowledgementErrorMessageKey].ToString(), segName + " OnlyOneName - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst2[detailedErrorMessageKey].ToString(), segName + " OnlyOneName - detailed error");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_NoMiddleName()
        {
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("~MIDDLENAME", string.Empty));
            Assert.IsNull(tst1, "NoMiddleName PID");

            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("~MIDDLENAME", string.Empty));
            Assert.IsNull(tst2, "NoMiddleName MRG");
        }

        [Test]
        [ExpectedException(typeof(ArgumentNullException))]
        public void ValidateVbecsRequiredMessageContent_NullHL7Message()
        {
            va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(null);
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_MissingSegment()
        {
            segName = PID;
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("PID", "POO"));
            Assert.AreEqual(2, tst1.Count, segName + "-error count - MissingSegment");
            Assert.AreEqual(patientNameNotFound, tst1[acknowledgementErrorMessageKey].ToString(), segName + " MissingSegment - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst1[detailedErrorMessageKey].ToString(), segName + " MissingSegment - detailed error");

            segName = MRG;
            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("MRG", "POO").Replace("PID", "POO"));   //(CPRS_MESSAGE + "MRG" + CARRIAGE_RETURN);
            Assert.AreEqual(2, tst2.Count, segName + "-error count - MissingSegment");
            Assert.AreEqual(patientNameNotFound, tst2[acknowledgementErrorMessageKey].ToString(), segName + " MissingSegment - acknowledgment");
            Assert.AreEqual(patientNameNotFound, tst2[detailedErrorMessageKey].ToString(), segName + " MissingSegment - detailed error");
        }

        [Test]
        public void ValidateVbecsRequiredMessageContent_AllNamesTooLong()
        {
            var midmax = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientMiddleNameMaximumLength", typeof(int));
            var firstmax = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientFirstNameMaximumLength", typeof(int));
            var lastmax = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientLastNameMaximumLength", typeof(int));
            var fullmax = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientFullNameMaximumLength", typeof(int));

            segName = "PID";
            Hashtable tst1 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(CPRS_MESSAGE.Replace("MIDDLENAME", "M".PadRight(midmax + 1, 'M')).Replace("LASTNAME","L".PadRight(lastmax+1,'L')).Replace("FIRSTNAME", "F".PadRight(firstmax+1,'L')));
            StringBuilder errChk = new StringBuilder();
            errChk.Append(string.Format(NameComponentTooBig, "Last", lastmax+1, lastmax));
            errChk.Append("\n");
            errChk.Append(string.Format(NameComponentTooBig, "First", firstmax+1, firstmax));
            errChk.Append("\n");
            errChk.Append(string.Format(NameComponentTooBig, "Middle", midmax + 1, midmax));
            errChk.Append(string.Format(FullNameTooBig, lastmax + 1 + firstmax + 1 + midmax + 1 + 2, fullmax));

            Assert.AreEqual(2, tst1.Count, "All names too long in " + segName + " - error count");
            Assert.AreEqual(patientNameFieldSizeNotSupported, tst1[acknowledgementErrorMessageKey].ToString(), "All names too long in " + segName + "  - acknowledgment");
            Assert.AreEqual(errChk.ToString(), tst1[detailedErrorMessageKey].ToString(), "All names too long in " + segName + "  - detailed error");

            segName = "MRG";
            Hashtable tst2 = va.med.vbecs.HL7Server.Server.ValidateVbecsRequiredMessageContent(PATIENT_MERGE_MESSAGE.Replace("MIDDLENAME", "M".PadRight(midmax + 1, 'M')).Replace("LASTNAME","L".PadRight(lastmax+1,'L')).Replace("FIRSTNAME", "F".PadRight(firstmax+1,'L')));
            Assert.AreEqual(2, tst2.Count, "All names too long in " + segName + " - error count");
            Assert.AreEqual(patientNameFieldSizeNotSupported, tst2[acknowledgementErrorMessageKey].ToString(), "All names too long in " + segName + "  - acknowledgment");
            Assert.AreEqual(errChk.ToString(), tst2[detailedErrorMessageKey].ToString(), "All names too long in " + segName + "  - detailed error");

        }

    }
}
#endif